/**
 *
 * \file        uart_console.c
 *
 * \brief       Implementation of a console works over a serial port.
 *
 * \author      Pete McCormick (loosely based on uart_console.c in Cajita project)
 *
 * \date        12/5/2007
 *
 * \note        This is a generic file.  There should be NO product-specific
 *              nor processor-specific stuff in here.
 *
 */

////////////////////////////////////////////////////////////////////////////////

#include "console.h"
#include "os.h"
#include "string_utils.h"
#include <string.h>
#include "leds.h"
#include "hardware.h"

////////////////////////////////////////////////////////////////////////////////

UARTCONSOLE UartConsole;

////////////////////////////////////////////////////////////////////////////////

/**
 * \author      Pete McCormick
 *
 * \date        3/10/2008
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Unused for now - needed if we ever implement
 *              XGET, XPUT, or SYSTEM commands.
 *
 * \param       void
 *
 */
void UartConsoleCaptureInput(void)
{
}

/**
 * \author      Pete McCormick
 *
 * \date        3/10/2008
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Unused for now - needed if we ever implement
 *              XGET, XPUT, or SYSTEM commands.
 *
 * \param       void
 *
 */
void UartConsoleReleaseInput(void)
{
}

// need empty function for devices without console
void UartNoConsoleTx(UINT8 *data, UINT32 count, UINT32 inst)
{
    ;
}

/**
 * \author      Pete McCormick
 *
 * \date        3/10/2008
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Sets up structures used for the UART console.
 *
 * \param       void
 *
 * \note        Runs once at startup time.
 *
 */
void UartConsoleInit(void)
{
    CMDINPUT * pCmdInput = &UartConsole.cmdInput;
    UINT32 bps = 115200;     // this will be configurable later
    UINT32 uart = HwGetConsoleUart();

    // needed when actually parsing the command
    pCmdInput->command.pCmdInput = pCmdInput;

    if (uart == NO_CONSOLE_UART_PRESENT) {
        pCmdInput->pSendFunc = UartNoConsoleTx;
    }
    else {
        pCmdInput->pSendFunc = UartConsoleTx;
    }
    pCmdInput->sendParam = uart;
    pCmdInput->pCaptureFunc = UartConsoleCaptureInput;
    pCmdInput->pReleaseFunc = UartConsoleReleaseInput;
    pCmdInput->cmdSource = FROM_UART_CONSOLE;
    pCmdInput->printLock = 0;
    pCmdInput->captured  = 0;
    pCmdInput->TjiPrintLock = 0;
    pCmdInput->consoleLock = 0;
    pCmdInput->pPassto = NULL;


    CmdInputAddToList(pCmdInput);

    HwUartSetBaudRate(uart, bps, 1);
}
/**
 * \author      Andrew Salmon
 *
 * \date        02/19/2009
 *
 * \return      UINT8
 *
 * \retval      true if console was initialized
 *
 * \brief       query the cmdsource for console initialization.
 *
 * \param       void
 */
UINT8 IsUartConsoleInit(void)
{
   CMDINPUT * pCmdInput = &UartConsole.cmdInput;

   return (pCmdInput->cmdSource == FROM_UART_CONSOLE);
}

/**
 * \author      Pete McCormick
 *
 * \date        3/10/2008
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       (Upper-level) Interrupt service routine for
 *              the UART console.
 *
 * \param       inst - which UART
 * \param       gotBreak : 1 if we received a "break", 0 otherwise
 * \param       rxReady : 1 if we received at least 1 character
 * \param       newByte : the character just received
 * \param       txEmpty : 1 if the transmitter is done sending something
 * \warning     Runs from interrupt context.
 *
 */
UINT32 UartConsoleIsr(UINT32 inst, BOOL gotBreak, BOOL rxReady, UINT8 newByte, BOOL txEmpty)
{
    CMDINPUT * pCmdInput = &UartConsole.cmdInput;
    UINT32 flags = 0;

    if(rxReady)
    {
        flags |= DmConsoleBufferInput(pCmdInput, newByte);
    }


    if(txEmpty)
    {
        if(UartConsole.msgLen--)
        {
            flags |= UartConsole.pTxBuffer[UartConsole.charIdx++];
            return flags;       // not done, more to send
        }
        else
            UartConsole.TxDone = TRUE;
    }

    // nothing to tx - disable int
    flags |= OS_ISR_FLAG_NO_MORE_DATA;
    return flags;
}

/**
 * \author      Pete McCormick
 *
 * \date        3/10/2008
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Displays the initial banner message on the console
 *              when a console session begins.
 *
 * \param       void
 *
 */
void UartConsoleDisplayWelcome(void)
{
    DmConsoleDisplayWelcome(&UartConsole.cmdInput);
}


void UartConsoleMsg(const char *msg)
{
    UartConsoleTx((unsigned char*)msg, strlen(msg), HwGetConsoleUart());
}

